/*
MD_MAX72xx - Library for using a MAX7219/7221 LED matrix controller

See header file for comments

This file contains methods that work with the fonts and characters defined in the library

Copyright (C) 2012-14 Marco Colli. All rights reserved.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
#include <Arduino.h>
#include "MD_MAX72xx.h"
#include "MD_MAX72xx_lib.h"

/**
 * \file
 * \brief Implements font definition and methods
 */

#if USE_LOCAL_FONT
// Local font handling functions if the option is enabled

void MD_MAX72XX::setFontInfoDefault(void)
// Set the defaults for the info block compatible with version 0 of the file
{
  _fontInfo.version = 0;
  _fontInfo.height = 8;
  _fontInfo.widthMax = 0;
  _fontInfo.firstASCII = 0;
  _fontInfo.lastASCII = 255;
  _fontInfo.dataOffset = 0;
}

void MD_MAX72XX::loadFontInfo(void)
{
  uint8_t c;
  uint16_t offset = 0;
  
  setFontInfoDefault();

  if (_fontData != nullptr)
  {
    PRINTS("\nLoading font info");
    // Read the first character. If this is not the file type indicator
    // then we have a version 0 file and the defaults are ok, otherwise 
    // read the font info from the data table. 
    c = pgm_read_byte(_fontData + offset++);
    if (c == FONT_FILE_INDICATOR)
    {
      c = pgm_read_byte(_fontData + offset++);  // read the version number
      switch (c)
      {
        case 1:
          _fontInfo.firstASCII = pgm_read_byte(_fontData + offset++);
          _fontInfo.lastASCII  = pgm_read_byte(_fontData + offset++);
          _fontInfo.height     = pgm_read_byte(_fontData + offset++);
          break;
        
        case 0:
        default:
          // nothing to do, use the library defaults
          break;
      }
      _fontInfo.dataOffset = offset;
    }
    // these always set
    _fontInfo.widthMax = getFontWidth();
  }
}

uint8_t MD_MAX72XX::getFontWidth(void)
{
  uint8_t   max = 0;
  uint8_t   charWidth;
  uint16_t  offset = _fontInfo.dataOffset;

  PRINTS("\nFinding max font width");
  if (_fontData != nullptr)
  {
    for (uint16_t i = _fontInfo.firstASCII; i <= _fontInfo.lastASCII; i++)
    {
      charWidth = pgm_read_byte(_fontData + offset);
      /*
      PRINT("\nASCII '", i);
      PRINT("' offset ", offset);
      PRINT("' width ", charWidth);
      */
      if (charWidth > max)
      {
        max = charWidth;
        PRINT(":", max);
      }
      offset += charWidth;  // skip character data
      offset++; // skip to size byte
    }
  }
  PRINT(" max ", max);

  return(max);
}

int16_t MD_MAX72XX::getFontCharOffset(uint8_t c)
{
  int16_t  offset = _fontInfo.dataOffset;

  PRINT("\nfontOffset ASCII ", c);

  if (c < _fontInfo.firstASCII || c > _fontInfo.lastASCII)
    offset = -1;
  else
  {
    for (uint8_t i=_fontInfo.firstASCII; i<c; i++)
    {
      PRINTS(".");
      offset += pgm_read_byte(_fontData+offset);
      offset++; // skip size byte we used above
    }

    PRINT(" searched offset ", offset);
    return(offset);
  }
}

bool MD_MAX72XX::setFont(fontType_t *f)
{
  if (f != _fontData) // we actually have a change to process
  {
    _fontData = (f == nullptr ? _sysfont_var : f);
    loadFontInfo();
  }

  return(true);
}

uint8_t MD_MAX72XX::getChar(uint8_t c, uint8_t size, uint8_t *buf)
{
  PRINT("\ngetChar: '", (char)c);
  PRINT("' ASC ", c);
  PRINT(" - bufsize ", size);

  if (buf == nullptr)
    return(0);

  int16_t offset = getFontCharOffset(c);
  if (offset == -1)
  {
    memset(buf, 0, size);
    size = 0;
  }
  else
  {
    size = min(size, pgm_read_byte(_fontData+offset));

    offset++; // skip the size byte

    for (uint8_t i=0; i<size; i++)
      *buf++ = pgm_read_byte(_fontData+offset+i);
  }
  
  return(size);
}

uint8_t MD_MAX72XX::setChar(uint16_t col, uint8_t c)
{
  PRINT("\nsetChar: '", c);
  PRINT("' column ", col);
  boolean b = _updateEnabled;
  uint8_t size;

  int16_t offset = getFontCharOffset(c);
  if (offset == -1)
    return(0);

  size = pgm_read_byte(_fontData+offset);

  offset++; // skip the size byte

  _updateEnabled = false;
  for (int8_t i=0; i<size; i++)
  {
    uint8_t colData = pgm_read_byte(_fontData+offset+i);
    setColumn(col--, colData);
  }
  _updateEnabled = b;

  if (_updateEnabled) flushBufferAll();

  return(size);
}

// Standard font - variable spacing
MD_MAX72XX::fontType_t PROGMEM _sysfont_var[] =
{
  'F', 1, 0, 255, 8,
  0,  // 0 - 'Empty Cell'
  5, 0x3e, 0x5b, 0x4f, 0x5b, 0x3e,  // 1 - 'Sad Smiley'
  5, 0x3e, 0x6b, 0x4f, 0x6b, 0x3e,  // 2 - 'Happy Smiley'
  5, 0x1c, 0x3e, 0x7c, 0x3e, 0x1c,  // 3 - 'Heart'
  5, 0x18, 0x3c, 0x7e, 0x3c, 0x18,  // 4 - 'Diamond'
  5, 0x1c, 0x57, 0x7d, 0x57, 0x1c,  // 5 - 'Clubs'
  5, 0x1c, 0x5e, 0x7f, 0x5e, 0x1c,  // 6 - 'Spades'
  4, 0x00, 0x18, 0x3c, 0x18,  // 7 - 'Bullet Point'
  5, 0xff, 0xe7, 0xc3, 0xe7, 0xff,  // 8 - 'Rev Bullet Point'
  4, 0x00, 0x18, 0x24, 0x18,  // 9 - 'Hollow Bullet Point'
  5, 0xff, 0xe7, 0xdb, 0xe7, 0xff,  // 10 - 'Rev Hollow BP'
  5, 0x30, 0x48, 0x3a, 0x06, 0x0e,  // 11 - 'Male'
  5, 0x26, 0x29, 0x79, 0x29, 0x26,  // 12 - 'Female'
  5, 0x40, 0x7f, 0x05, 0x05, 0x07,  // 13 - 'Music Note 1'
  5, 0x40, 0x7f, 0x05, 0x25, 0x3f,  // 14 - 'Music Note 2'
  5, 0x5a, 0x3c, 0xe7, 0x3c, 0x5a,  // 15 - 'Snowflake'
  5, 0x7f, 0x3e, 0x1c, 0x1c, 0x08,  // 16 - 'Right Pointer'
  5, 0x08, 0x1c, 0x1c, 0x3e, 0x7f,  // 17 - 'Left Pointer'
  5, 0x14, 0x22, 0x7f, 0x22, 0x14,  // 18 - 'UpDown Arrows'
  5, 0x5f, 0x5f, 0x00, 0x5f, 0x5f,  // 19 - 'Double Exclamation'
  5, 0x06, 0x09, 0x7f, 0x01, 0x7f,  // 20 - 'Paragraph Mark'
  4, 0x66, 0x89, 0x95, 0x6a,  // 21 - 'Section Mark'
  5, 0x60, 0x60, 0x60, 0x60, 0x60,  // 22 - 'Double Underline'
  5, 0x94, 0xa2, 0xff, 0xa2, 0x94,  // 23 - 'UpDown Underlined'
  5, 0x08, 0x04, 0x7e, 0x04, 0x08,  // 24 - 'Up Arrow'
  5, 0x10, 0x20, 0x7e, 0x20, 0x10,  // 25 - 'Down Arrow'
  5, 0x08, 0x08, 0x2a, 0x1c, 0x08,  // 26 - 'Right Arrow'
  5, 0x08, 0x1c, 0x2a, 0x08, 0x08,  // 27 - 'Left Arrow'
  5, 0x1e, 0x10, 0x10, 0x10, 0x10,  // 28 - 'Angled'
  5, 0x0c, 0x1e, 0x0c, 0x1e, 0x0c,  // 29 - 'Squashed #'
  5, 0x30, 0x38, 0x3e, 0x38, 0x30,  // 30 - 'Up Pointer'
  5, 0x06, 0x0e, 0x3e, 0x0e, 0x06,  // 31 - 'Down Pointer'
  2, 0x00, 0x00,  // 32 - 'Space'
  1, 0x5f,  // 33 - '!'
  3, 0x07, 0x00, 0x07,  // 34 - '"'
  5, 0x14, 0x7f, 0x14, 0x7f, 0x14,  // 35 - '#'
  5, 0x24, 0x2a, 0x7f, 0x2a, 0x12,  // 36 - '$'
  5, 0x23, 0x13, 0x08, 0x64, 0x62,  // 37 - '%'
  5, 0x36, 0x49, 0x56, 0x20, 0x50,  // 38 - '&'
  3, 0x08, 0x07, 0x03,  // 39 - '''
  3, 0x1c, 0x22, 0x41,  // 40 - '('
  3, 0x41, 0x22, 0x1c,  // 41 - ')'
  5, 0x2a, 0x1c, 0x7f, 0x1c, 0x2a,  // 42 - '*'
  5, 0x08, 0x08, 0x3e, 0x08, 0x08,  // 43 - '+'
  3, 0x80, 0x70, 0x30,  // 44 - ','
  5, 0x08, 0x08, 0x08, 0x08, 0x08,  // 45 - '-'
  2, 0x60, 0x60,  // 46 - '.'
  5, 0x20, 0x10, 0x08, 0x04, 0x02,  // 47 - '/'
  5, 0x3e, 0x51, 0x49, 0x45, 0x3e,  // 48 - '0'
  3, 0x42, 0x7f, 0x40,  // 49 - '1'
  5, 0x72, 0x49, 0x49, 0x49, 0x46,  // 50 - '2'
  5, 0x21, 0x41, 0x49, 0x4d, 0x33,  // 51 - '3'
  5, 0x18, 0x14, 0x12, 0x7f, 0x10,  // 52 - '4'
  5, 0x27, 0x45, 0x45, 0x45, 0x39,  // 53 - '5'
  5, 0x3c, 0x4a, 0x49, 0x49, 0x31,  // 54 - '6'
  5, 0x41, 0x21, 0x11, 0x09, 0x07,  // 55 - '7'
  5, 0x36, 0x49, 0x49, 0x49, 0x36,  // 56 - '8'
  5, 0x46, 0x49, 0x49, 0x29, 0x1e,  // 57 - '9'
  1, 0x14,  // 58 - ':'
  2, 0x80, 0x68,  // 59 - ';'
  4, 0x08, 0x14, 0x22, 0x41,  // 60 - '<'
  5, 0x14, 0x14, 0x14, 0x14, 0x14,  // 61 - '='
  4, 0x41, 0x22, 0x14, 0x08,  // 62 - '>'
  5, 0x02, 0x01, 0x59, 0x09, 0x06,  // 63 - '?'
  5, 0x3e, 0x41, 0x5d, 0x59, 0x4e,  // 64 - '@'
  5, 0x7c, 0x12, 0x11, 0x12, 0x7c,  // 65 - 'A'
  5, 0x7f, 0x49, 0x49, 0x49, 0x36,  // 66 - 'B'
  5, 0x3e, 0x41, 0x41, 0x41, 0x22,  // 67 - 'C'
  5, 0x7f, 0x41, 0x41, 0x41, 0x3e,  // 68 - 'D'
  5, 0x7f, 0x49, 0x49, 0x49, 0x41,  // 69 - 'E'
  5, 0x7f, 0x09, 0x09, 0x09, 0x01,  // 70 - 'F'
  5, 0x3e, 0x41, 0x41, 0x51, 0x73,  // 71 - 'G'
  5, 0x7f, 0x08, 0x08, 0x08, 0x7f,  // 72 - 'H'
  3, 0x41, 0x7f, 0x41,  // 73 - 'I'
  5, 0x20, 0x40, 0x41, 0x3f, 0x01,  // 74 - 'J'
  5, 0x7f, 0x08, 0x14, 0x22, 0x41,  // 75 - 'K'
  5, 0x7f, 0x40, 0x40, 0x40, 0x40,  // 76 - 'L'
  5, 0x7f, 0x02, 0x1c, 0x02, 0x7f,  // 77 - 'M'
  5, 0x7f, 0x04, 0x08, 0x10, 0x7f,  // 78 - 'N'
  5, 0x3e, 0x41, 0x41, 0x41, 0x3e,  // 79 - 'O'
  5, 0x7f, 0x09, 0x09, 0x09, 0x06,  // 80 - 'P'
  5, 0x3e, 0x41, 0x51, 0x21, 0x5e,  // 81 - 'Q'
  5, 0x7f, 0x09, 0x19, 0x29, 0x46,  // 82 - 'R'
  5, 0x26, 0x49, 0x49, 0x49, 0x32,  // 83 - 'S'
  5, 0x03, 0x01, 0x7f, 0x01, 0x03,  // 84 - 'T'
  5, 0x3f, 0x40, 0x40, 0x40, 0x3f,  // 85 - 'U'
  5, 0x1f, 0x20, 0x40, 0x20, 0x1f,  // 86 - 'V'
  5, 0x3f, 0x40, 0x38, 0x40, 0x3f,  // 87 - 'W'
  5, 0x63, 0x14, 0x08, 0x14, 0x63,  // 88 - 'X'
  5, 0x03, 0x04, 0x78, 0x04, 0x03,  // 89 - 'Y'
  5, 0x61, 0x59, 0x49, 0x4d, 0x43,  // 90 - 'Z'
  3, 0x7f, 0x41, 0x41,  // 91 - '['
  5, 0x02, 0x04, 0x08, 0x10, 0x20,  // 92 - '\'
  3, 0x41, 0x41, 0x7f,  // 93 - ']'
  5, 0x04, 0x02, 0x01, 0x02, 0x04,  // 94 - '^'
  5, 0x40, 0x40, 0x40, 0x40, 0x40,  // 95 - '_'
  3, 0x03, 0x07, 0x08,  // 96 - '`'
  5, 0x20, 0x54, 0x54, 0x78, 0x40,  // 97 - 'a'
  5, 0x7f, 0x28, 0x44, 0x44, 0x38,  // 98 - 'b'
  5, 0x38, 0x44, 0x44, 0x44, 0x28,  // 99 - 'c'
  5, 0x38, 0x44, 0x44, 0x28, 0x7f,  // 100 - 'd'
  5, 0x38, 0x54, 0x54, 0x54, 0x18,  // 101 - 'e'
  4, 0x08, 0x7e, 0x09, 0x02,  // 102 - 'f'
  5, 0x18, 0xa4, 0xa4, 0x9c, 0x78,  // 103 - 'g'
  5, 0x7f, 0x08, 0x04, 0x04, 0x78,  // 104 - 'h'
  3, 0x44, 0x7d, 0x40,  // 105 - 'i'
  4, 0x40, 0x80, 0x80, 0x7a,  // 106 - 'j'
  4, 0x7f, 0x10, 0x28, 0x44,  // 107 - 'k'
  3, 0x41, 0x7f, 0x40,  // 108 - 'l'
  5, 0x7c, 0x04, 0x78, 0x04, 0x78,  // 109 - 'm'
  5, 0x7c, 0x08, 0x04, 0x04, 0x78,  // 110 - 'n'
  5, 0x38, 0x44, 0x44, 0x44, 0x38,  // 111 - 'o'
  5, 0xfc, 0x18, 0x24, 0x24, 0x18,  // 112 - 'p'
  5, 0x18, 0x24, 0x24, 0x18, 0xfc,  // 113 - 'q'
  5, 0x7c, 0x08, 0x04, 0x04, 0x08,  // 114 - 'r'
  5, 0x48, 0x54, 0x54, 0x54, 0x24,  // 115 - 's'
  4, 0x04, 0x3f, 0x44, 0x24,  // 116 - 't'
  5, 0x3c, 0x40, 0x40, 0x20, 0x7c,  // 117 - 'u'
  5, 0x1c, 0x20, 0x40, 0x20, 0x1c,  // 118 - 'v'
  5, 0x3c, 0x40, 0x30, 0x40, 0x3c,  // 119 - 'w'
  5, 0x44, 0x28, 0x10, 0x28, 0x44,  // 120 - 'x'
  5, 0x4c, 0x90, 0x90, 0x90, 0x7c,  // 121 - 'y'
  5, 0x44, 0x64, 0x54, 0x4c, 0x44,  // 122 - 'z'
  3, 0x08, 0x36, 0x41,  // 123 - '{'
  1, 0x77,  // 124 - '|'
  3, 0x41, 0x36, 0x08,  // 125 - '}'
  5, 0x02, 0x01, 0x02, 0x04, 0x02,  // 126 - '~'
  5, 0x3c, 0x26, 0x23, 0x26, 0x3c,  // 127 - 'Hollow Up Arrow'
  5, 0x1e, 0xa1, 0xa1, 0x61, 0x12,  // 128 - 'C sedilla'
  5, 0x38, 0x42, 0x40, 0x22, 0x78,  // 129 - 'u umlaut'
  5, 0x38, 0x54, 0x54, 0x55, 0x59,  // 130 - 'e acute'
  5, 0x21, 0x55, 0x55, 0x79, 0x41,  // 131 - 'a accent'
  5, 0x21, 0x54, 0x54, 0x78, 0x41,  // 132 - 'a umlaut'
  5, 0x21, 0x55, 0x54, 0x78, 0x40,  // 133 - 'a grave'
  5, 0x20, 0x54, 0x55, 0x79, 0x40,  // 134 - 'a acute'
  5, 0x18, 0x3c, 0xa4, 0xe4, 0x24,  // 135 - 'c sedilla'
  5, 0x39, 0x55, 0x55, 0x55, 0x59,  // 136 - 'e accent'
  5, 0x38, 0x55, 0x54, 0x55, 0x58,  // 137 - 'e umlaut'
  5, 0x39, 0x55, 0x54, 0x54, 0x58,  // 138 - 'e grave'
  3, 0x45, 0x7c, 0x41,  // 139 - 'i umlaut'
  4, 0x02, 0x45, 0x7d, 0x42,  // 140 - 'i hat'
  4, 0x01, 0x45, 0x7c, 0x40,  // 141 - 'i grave'
  5, 0xf0, 0x29, 0x24, 0x29, 0xf0,  // 142 - 'A umlaut'
  5, 0xf0, 0x28, 0x25, 0x28, 0xf0,  // 143 - 'A dot'
  4, 0x7c, 0x54, 0x55, 0x45,  // 144 - 'E grave'
  7, 0x20, 0x54, 0x54, 0x7c, 0x54, 0x54, 0x08,  // 145 - 'ae'
  6, 0x7c, 0x0a, 0x09, 0x7f, 0x49, 0x49,  // 146 - 'AE'
  5, 0x32, 0x49, 0x49, 0x49, 0x32,  // 147 - 'o hat'
  5, 0x30, 0x4a, 0x48, 0x4a, 0x30,  // 148 - 'o umlaut'
  5, 0x32, 0x4a, 0x48, 0x48, 0x30,  // 149 - 'o grave'
  5, 0x3a, 0x41, 0x41, 0x21, 0x7a,  // 150 - 'u hat'
  5, 0x3a, 0x42, 0x40, 0x20, 0x78,  // 151 - 'u grave'
  4, 0x9d, 0xa0, 0xa0, 0x7d,  // 152 - 'y umlaut'
  5, 0x38, 0x45, 0x44, 0x45, 0x38,  // 153 - 'O umlaut'
  5, 0x3c, 0x41, 0x40, 0x41, 0x3c,  // 154 - 'U umlaut'
  5, 0x3c, 0x24, 0xff, 0x24, 0x24,  // 155 - 'Cents'
  5, 0x48, 0x7e, 0x49, 0x43, 0x66,  // 156 - 'Pounds'
  5, 0x2b, 0x2f, 0xfc, 0x2f, 0x2b,  // 157 - 'Yen'
  5, 0xff, 0x09, 0x29, 0xf6, 0x20,  // 158 - 'R +'
  5, 0xc0, 0x88, 0x7e, 0x09, 0x03,  // 159 - 'f notation'
  5, 0x20, 0x54, 0x54, 0x79, 0x41,  // 160 - 'a acute'
  3, 0x44, 0x7d, 0x41,  // 161 - 'i acute'
  5, 0x30, 0x48, 0x48, 0x4a, 0x32,  // 162 - 'o acute'
  5, 0x38, 0x40, 0x40, 0x22, 0x7a,  // 163 - 'u acute'
  4, 0x7a, 0x0a, 0x0a, 0x72,  // 164 - 'n accent'
  5, 0x7d, 0x0d, 0x19, 0x31, 0x7d,  // 165 - 'N accent'
  5, 0x26, 0x29, 0x29, 0x2f, 0x28,  // 166
  5, 0x26, 0x29, 0x29, 0x29, 0x26,  // 167
  5, 0x30, 0x48, 0x4d, 0x40, 0x20,  // 168 - 'Inverted ?'
  5, 0x38, 0x08, 0x08, 0x08, 0x08,  // 169 - 'LH top corner'
  5, 0x08, 0x08, 0x08, 0x08, 0x38,  // 170 - 'RH top corner'
  5, 0x2f, 0x10, 0xc8, 0xac, 0xba,  // 171 - '1/2'
  5, 0x2f, 0x10, 0x28, 0x34, 0xfa,  // 172 - '1/4'
  1, 0x7b,  // 173 - '| split'
  5, 0x08, 0x14, 0x2a, 0x14, 0x22,  // 174 - '<<'
  5, 0x22, 0x14, 0x2a, 0x14, 0x08,  // 175 - '>>'
  5, 0xaa, 0x00, 0x55, 0x00, 0xaa,  // 176 - '30% shading'
  5, 0xaa, 0x55, 0xaa, 0x55, 0xaa,  // 177 - '50% shading'
  5, 0x00, 0x00, 0x00, 0x00, 0xff,  // 178 - 'Right side'
  5, 0x10, 0x10, 0x10, 0x10, 0xff,  // 179 - 'Right T'
  5, 0x14, 0x14, 0x14, 0x14, 0xff,  // 180 - 'Right T double H'
  5, 0x10, 0x10, 0xff, 0x00, 0xff,  // 181 - 'Right T double V'
  5, 0x10, 0x10, 0xf0, 0x10, 0xf0,  // 182 - 'Top Right double V'
  5, 0x14, 0x14, 0x14, 0x14, 0xfc,  // 183 - 'Top Right double H'
  5, 0x14, 0x14, 0xf7, 0x00, 0xff,  // 184 - 'Right T double all'
  5, 0x00, 0x00, 0xff, 0x00, 0xff,  // 185 - 'Right side double'
  5, 0x14, 0x14, 0xf4, 0x04, 0xfc,  // 186 - 'Top Right double'
  5, 0x14, 0x14, 0x17, 0x10, 0x1f,  // 187 - 'Bot Right double'
  5, 0x10, 0x10, 0x1f, 0x10, 0x1f,  // 188 - 'Bot Right double V'
  5, 0x14, 0x14, 0x14, 0x14, 0x1f,  // 189 - 'Bot Right double H'
  5, 0x10, 0x10, 0x10, 0x10, 0xf0,  // 190 - 'Top Right'
  5, 0x00, 0x00, 0x00, 0x1f, 0x10,  // 191 - 'Bot Left'
  5, 0x10, 0x10, 0x10, 0x1f, 0x10,  // 192 - 'Bot T'
  5, 0x10, 0x10, 0x10, 0xf0, 0x10,  // 193 - 'Top T'
  5, 0x00, 0x00, 0x00, 0xff, 0x10,  // 194 - 'Left T'
  5, 0x10, 0x10, 0x10, 0x10, 0x10,  // 195 - 'Top side'
  5, 0x10, 0x10, 0x10, 0xff, 0x10,  // 196 - 'Center +'
  5, 0x00, 0x00, 0x00, 0xff, 0x14,  // 197 - 'Left side double H'
  5, 0x00, 0x00, 0xff, 0x00, 0xff,  // 198 - 'Left side double'
  5, 0x00, 0x00, 0x1f, 0x10, 0x17,  // 199 - 'Bot Left double V'
  5, 0x00, 0x00, 0xfc, 0x04, 0xf4,  // 200 - 'Top Left double V'
  5, 0x14, 0x14, 0x17, 0x10, 0x17,  // 201 - 'Bot T double'
  5, 0x14, 0x14, 0xf4, 0x04, 0xf4,  // 202 - 'Top T double'
  5, 0x00, 0x00, 0xff, 0x00, 0xf7,  // 203 - 'Left Side double spl'
  5, 0x14, 0x14, 0x14, 0x14, 0x14,  // 204 - 'Center double'
  5, 0x14, 0x14, 0xf7, 0x00, 0xf7,  // 205 - 'Center + double'
  5, 0x14, 0x14, 0x14, 0x17, 0x14,  // 206 - 'Bot T double H'
  5, 0x10, 0x10, 0x1f, 0x10, 0x1f,  // 207 - 'Bot Right double V'
  5, 0x14, 0x14, 0x14, 0xf4, 0x14,  // 208 - 'Top T double H'
  5, 0x10, 0x10, 0xf0, 0x10, 0xf0,  // 209 - 'Top Right double V'
  5, 0x00, 0x00, 0x1f, 0x10, 0x1f,  // 210 - 'Bot Left double V'
  5, 0x00, 0x00, 0x00, 0x1f, 0x14,  // 211 - 'Bot Right double H'
  5, 0x00, 0x00, 0x00, 0xfc, 0x14,  // 212 - 'Top Right double H'
  5, 0x00, 0x00, 0xf0, 0x10, 0xf0,  // 213 - 'Top Right double V'
  5, 0x10, 0x10, 0xff, 0x10, 0xff,  // 214 - 'Center + double V'
  5, 0x14, 0x14, 0x14, 0xff, 0x14,  // 215 - 'Center + double H'
  5, 0x10, 0x10, 0x10, 0x10, 0x1f,  // 216 - 'Bot Right'
  5, 0x00, 0x00, 0x00, 0xf0, 0x10,  // 217 - 'Top Left'
  5, 0xff, 0xff, 0xff, 0xff, 0xff,  // 218 - 'Full Block'
  5, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,  // 219 - 'Half Block Bottom'
  3, 0xff, 0xff, 0xff,  // 220 - 'Half Block LHS'
  5, 0x00, 0x00, 0x00, 0xff, 0xff,  // 221 - 'Half Block RHS'
  5, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,  // 222 - 'Half Block Top'
  5, 0x38, 0x44, 0x44, 0x38, 0x44,  // 223 - 'Alpha'
  5, 0x7c, 0x2a, 0x2a, 0x3e, 0x14,  // 224 - 'Beta'
  5, 0x7e, 0x02, 0x02, 0x06, 0x06,  // 225 - 'Gamma'
  5, 0x02, 0x7e, 0x02, 0x7e, 0x02,  // 226 - 'Pi'
  5, 0x63, 0x55, 0x49, 0x41, 0x63,  // 227 - 'Sigma'
  5, 0x38, 0x44, 0x44, 0x3c, 0x04,  // 228 - 'Theta'
  5, 0x40, 0x7e, 0x20, 0x1e, 0x20,  // 229 - 'mu'
  5, 0x06, 0x02, 0x7e, 0x02, 0x02,  // 230 - 'Tau'
  5, 0x99, 0xa5, 0xe7, 0xa5, 0x99,  // 231
  5, 0x1c, 0x2a, 0x49, 0x2a, 0x1c,  // 232
  5, 0x4c, 0x72, 0x01, 0x72, 0x4c,  // 233
  5, 0x30, 0x4a, 0x4d, 0x4d, 0x30,  // 234
  5, 0x30, 0x48, 0x78, 0x48, 0x30,  // 235
  5, 0xbc, 0x62, 0x5a, 0x46, 0x3d,  // 236 - 'Zero Slashed'
  4, 0x3e, 0x49, 0x49, 0x49,  // 237
  5, 0x7e, 0x01, 0x01, 0x01, 0x7e,  // 238
  5, 0x2a, 0x2a, 0x2a, 0x2a, 0x2a,  // 239 - '3 Bar Equals'
  5, 0x44, 0x44, 0x5f, 0x44, 0x44,  // 240 - '+/-'
  5, 0x40, 0x51, 0x4a, 0x44, 0x40,  // 241 - '>='
  5, 0x40, 0x44, 0x4a, 0x51, 0x40,  // 242 - '<='
  5, 0x00, 0x00, 0xff, 0x01, 0x03,  // 243 - 'Top of Integral'
  3, 0xe0, 0x80, 0xff,  // 244 - 'Bot of Integral'
  5, 0x08, 0x08, 0x6b, 0x6b, 0x08,  // 245 - 'Divide'
  5, 0x36, 0x12, 0x36, 0x24, 0x36,  // 246 - 'Wavy ='
  5, 0x06, 0x0f, 0x09, 0x0f, 0x06,  // 247 - 'Degree'
  4, 0x00, 0x00, 0x18, 0x18,  // 248 - 'Math Product'
  4, 0x00, 0x00, 0x10, 0x10,  // 249 - 'Short Dash'
  5, 0x30, 0x40, 0xff, 0x01, 0x01,  // 250 - 'Square Root'
  5, 0x00, 0x1f, 0x01, 0x01, 0x1e,  // 251 - 'Superscript n'
  5, 0x00, 0x19, 0x1d, 0x17, 0x12,  // 252 - 'Superscript 2'
  5, 0x00, 0x3c, 0x3c, 0x3c, 0x3c,  // 253 - 'Centered Square'
  5, 0xff, 0x81, 0x81, 0x81, 0xff,  // 254 - 'Full Frame'
  5, 0xff, 0xff, 0xff, 0xff, 0xff,  // 255 - 'Full Block'
};

#endif //INCLUDE_LOCAL_FONT

